
/*******************************************************************************
*
*                    TWO DIMENSIONAL ELASTIC MEMBRANE
*
*                              C++ Code
*                            Version: January 2009
*      Author: Arash Zamani, PhD from Amirkabir University of Technology
*        
*          This code has been tested with Microsoft Visual C++ 2008
********************************************************************************
*                                                                            
*                       Introducing global variables                         
*                                                                            
* nx=number of divisions in x direction                                    
* ny=number of divisions in y direction                                   
* nr=number of divisions in radial direction for an elliptic domain       
* nc=number of divisions in angular direction for the first layer of       
*    elements around the ellipse center.                                   
* npe=number of nodes per element                                                      
* nb=bandwith                                                               
* ne=number of elements                                                     
* nn=number of nodes                                                        
* node=connectivity                                                          
* xv=vector of x values                                                     
* yv=vector of y values                                                      
* iconv=convection vector                                                   
* const1=determines whether to impose primary boundary conditions       
* const2=determines the value of primary boundary conditions                
* maxh=height of each column for skyline implementation                     
* gk=global stiffness matrix                                                
* p=the vector which is first used to store external forces and then         
*   to store the calculated primary variables from finite element model      
* Q=heat generation inside element (or pressure in membrane formulation).    
* q=heat flux to the boundary edges                                         
* h=the coefficient of convection heat transfer                             
* Tinf=temperature at infinity (surrounding media).                          
* kx,ky=thermal conductivities in x and y directions respectively            
*       (or tension in membrane formulation).                                
*                                                                            
*******************************************************************************/


#include <math.h>
#include <stdio.h> 
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <io.h>
#include <iostream>
#include <fstream>
#include <float.h>
using namespace std;

const long   nx=20, ny=32, nr=20, nc=6; 


//the variables for a rectangular domain and three node triangular element
//(rd_3nte).
const long   npe=3, nb=nx+2, ne=2*nx*ny, nn=(nx+1)*(ny+1);   

//the variables for a rectangular domain and six node triangular element
//(rd_6nte).
//const long   npe=6, nb=4*nx+3, ne=2*nx*ny, nn=(2*nx+1)*(2*ny+1);   

//the variables for a triangular domain and three node triangular element
//(td_3nte).
//const long   npe=3, nb=ny+2, ne=ny*ny, nn=(ny+1)*(ny+2)/2;  

//the variables for a triangular domain and six node triangular element
//(td_6nte).
//const long   npe=6, nb=4*ny+2, ne=ny*ny, nn=(2*ny+1)*(ny+1);

//the variables for an elliptic domain and three node triangular element
//(ed_3nte).
//const long   npe=3, nb=nc*(2*nr-3), ne=nc*(nr-1)*(nr-1), nn=nc*nr*(nr-1)/2+1;


long       node[npe*ne+1], iconv[nn+1], const1[nn+1],maxh[nn+2];
double     gk[nn*nb+1], p[nn+1], xv[nn+1], yv[nn+1], const2[nn+1];
double     Q[ne+1], h[nn+1], q[nn+1], Tinf[nn+1];
double     kx,ky;

void mesh_rd_3nte(double x0,double y0);
void mesh_rd_6nte(double x0,double y0);
void mesh_td_3nte(double x0,double y0);
void mesh_td_6nte(double x0,double y0);
void mesh_ed_3nte(double x0,double y0);
void bandwidth();
void core_3nte(char type);
void core_6nte(char type);
void boundary_sym_banded(long neqns,long nbw,double mat[],double rhs[],
			 long cond1[],double cond2[]);
void boundary_sym_skyline(long neqns,long nbw,double mat[],double rhs[],
			  long maxh[],long cond1[],double cond2[]);
void solve_gauss_sym_banded(long neqns,long nbw,double mat[],double rhs[]);
void solve_LUdecom_sym_skyline(long neqns,double mat[],double rhs[],long maxh[]);
void file_out();


void main(void)
{
    char type='b'; //specifies whether to choose the banded or skyline form.
                   //the value 'b' refers to banded form.
                   //the value 's' refers to skyline form.

    mesh_rd_3nte(5.0,8.0);

    switch(type){
        case 'b' :
            core_3nte(type);
            boundary_sym_banded(nn,nb,gk,p,const1,const2);
            solve_gauss_sym_banded(nn,nb,gk,p);
            break;
        case 's' :
            bandwidth();
            core_3nte(type);
            boundary_sym_skyline(nn,nb,gk,p,maxh,const1,const2);
            solve_LUdecom_sym_skyline(nn,gk,p,maxh);
            break;
    }

    file_out();
}


void mesh_rd_3nte(double x0,double y0)//generates the mesh for a rectangular 
                                      //domain by three node triangular elements
{
    long i,j,n,m,n1,n2,nx1,ny1;

    nx1=nx+1;
    ny1=ny+1;
    m=-6;
    for (j=1;j<=ny;j++){
        for (i=1;i<=nx;i++){
            m+=6;
            n=nx1*(j-1)+i;
            
            node[m+1]=n;
            node[m+2]=n+1;
            node[m+3]=n+nx+1;

            node[m+4]=n+1;
            node[m+5]=n+nx+2;
            node[m+6]=n+nx+1;
        }
    }

    for (i=1;i<=nx1;i++){
        for (j=1;j<=ny1;j++){
            n=nx1*(j-1)+i;
            xv[n]=double(i-1)*x0/double(nx);
            yv[n]=double(j-1)*y0/double(ny);
        }
    }

    for (i=1;i<=nn;i++){
        iconv[i]=0;
        const1[i]=0;
        const2[i]=0.0;
    }

    n=nx1*ny;
    for (i=1;i<=nx1;i++){
        n1=i;
        n2=i+n;
        const1[n1]=1;
        const1[n2]=1;

        const2[n1]=0.0;
        const2[n2]=0.0;
    }

    for (j=2;j<=ny;j++){
        n1=nx1*(j-1)+1;
        n2=nx1*j;
        const1[n1]=1;
        const1[n2]=1;

        const2[n1]=0.0;
        const2[n2]=0.0;
    }

    for(i=1;i<=ne;i++){
        Q[i]=10.0;
    }

	kx=1.0;ky=1.0;
}


void mesh_rd_6nte(double x0,double y0)//generates the mesh for a rectangular 
                                      //domain by six node triangular elements
{
    long i,j,n,m,n1,n2,nx1,ny1,ny2;
    
    nx1=2*nx+1;
    ny1=2*ny+1;
    m=-12;
    for (j=1;j<=ny;j++){
        for (i=1;i<=nx;i++){
            m+=12;
            n=2*nx1*(j-1)+2*i-1;
            
            node[m+1]=n;
            node[m+2]=n+2;
            node[m+3]=n+4*nx+2;
            node[m+4]=n+1;
            node[m+5]=n+2*nx+2;
            node[m+6]=n+2*nx+1;

            node[m+7]=n+2;
            node[m+8]=n+4*nx+4;
            node[m+9]=n+4*nx+2;
            node[m+10]=n+2*nx+3;
            node[m+11]=n+4*nx+3;
            node[m+12]=n+2*nx+2;
        }
    }

    for (i=1;i<=nx1;i++){
        for (j=1;j<=ny1;j++){
            n=nx1*(j-1)+i;
            xv[n]=double(i-1)*x0/double(2*nx);
            yv[n]=double(j-1)*y0/double(2*ny);
        }
    }

    for (i=1;i<=nn;i++){
        iconv[i]=0;
        const1[i]=0;
        const2[i]=0.0;
    }

    n=2*nx1*ny;
    for (i=1;i<=nx1;i++){
        n1=i;
        n2=i+n;
        const1[n1]=1;
        const1[n2]=1;

        const2[n1]=0.0;
        const2[n2]=0.0;
    }

    ny2=ny1-1;
    for (j=2;j<=ny2;j++){
        n1=nx1*(j-1)+1;
        n2=nx1*j;
        const1[n1]=1;
        const1[n2]=1;

        const2[n1]=0.0;
        const2[n2]=0.0;
    }

    for(i=1;i<=ne;i++){
        Q[i]=10.0;
    }

	kx=1.0;ky=1.0;
}


void mesh_td_3nte(double x0,double y0)//generates the mesh for a triangular 
                                      //domain by three node triangular elements
{
    long i,j,n,m,n1,n2,n3,ny1;

    ny1=ny+1;
    for (j=1;j<=ny;j++){
        for (i=1;i<=j-1;i++){
            m=3*((j-1)*(j-1)+2*(i-1));
            n=(j-1)*j/2+i;
            
            node[m+1]=n;
            node[m+2]=n+j+1;
            node[m+3]=n+j;

            node[m+4]=n;
            node[m+5]=n+1;
            node[m+6]=n+j+1;
        }
    }

    for(j=1;j<=ny;j++){
            n=j*(j+1)/2;
            m=j*j;

            node[3*(m-1)+1]=n;
            node[3*(m-1)+2]=n+j+1;
            node[3*(m-1)+3]=n+j;
    }


    for (j=1;j<=ny1;j++){
        for (i=1;i<=j;i++){
            n=(j-1)*j/2+i;
            xv[n]=double(i-1)*x0/double(ny);
            yv[n]=double(j-1)*y0/double(ny);
        }
    }

    for (i=1;i<=nn;i++){
        iconv[i]=0;
        const1[i]=0;
        const2[i]=0.0;
    }

    n=ny*ny1/2;
    for (j=1;j<=ny1;j++){
        n1=j*(j-1)/2+1;
        n2=j+n;
        n3=j*(j+1)/2;

        const1[n1]=1;
        const1[n2]=1;
        const1[n3]=1;

        const2[n1]=0.0;
        const2[n2]=0.0;
        const2[n3]=0.0;
    }

    for(i=1;i<=ne;i++){
        Q[i]=10.0;
    }

	kx=1.0;ky=1.0;
}


void mesh_td_6nte(double x0,double y0)//generates the mesh for a triangular 
                                      //domain by six node triangular elements
{
    long i,j,n,m,n1,n2,n3,ny1;

    ny1=2*ny+1;
    for (j=1;j<=ny;j++){
        for (i=1;i<=j-1;i++){
            m=6*((j-1)*(j-1)+2*(i-1));
            n=(j-1)*(2*j-1)+2*i-1;
            
            node[m+1]=n;
            node[m+2]=n+4*j+1;
            node[m+3]=n+4*j-1;
            node[m+4]=n+2*j;
            node[m+5]=n+4*j;
            node[m+6]=n+2*j-1;

            node[m+7]=n;
            node[m+8]=n+2;
            node[m+9]=n+4*j+1;
            node[m+10]=n+1;
            node[m+11]=n+2*j+1;
            node[m+12]=n+2*j;
        }
    }

    for(j=1;j<=ny;j++){
            n=(2*j-1)*j;
            m=j*j;

            node[6*(m-1)+1]=n;
            node[6*(m-1)+2]=n+4*j+1;
            node[6*(m-1)+3]=n+4*j-1;
            node[6*(m-1)+4]=n+2*j;
            node[6*(m-1)+5]=n+4*j;
            node[6*(m-1)+6]=n+2*j-1;
    }


    for (j=1;j<=ny1;j++){
        for (i=1;i<=j;i++){
            n=(j-1)*j/2+i;
            xv[n]=double(i-1)*x0/double(2*ny);
            yv[n]=double(j-1)*y0/double(2*ny);
        }
    }

    for (i=1;i<=nn;i++){
        iconv[i]=0;
        const1[i]=0;
        const2[i]=0.0;
    }

    n=ny*ny1;
    for (j=1;j<=ny1;j++){
        n1=j*(j-1)/2+1;
        n2=j+n;
        n3=j*(j+1)/2;

        const1[n1]=1;
        const1[n2]=1;
        const1[n3]=1;

        const2[n1]=0.0;
        const2[n2]=0.0;
        const2[n3]=0.0;
    }

    for(i=1;i<=ne;i++){
        Q[i]=10.0;
    }

	kx=1.0;ky=1.0;
}


void mesh_ed_3nte(double x0,double y0)//generates the mesh for an elliptic 
                                      //domain by three node triangular elements
{
    long   i,j,m,n,i1,j1,k1,m1,n1,nr1,ntheta;
    double pi=3.141592653589793;
    double sc,a,b,r,theta;

    nr1=nr-1;
    m=-3;
    n=0;
    i1=1;
    for(i=1;i<=nr1;i++){
        ntheta=nc*i;
        for(j=1;j<=ntheta;j++){
            m+=3;
            n++;

            if(j==1) node[m+1]=i1;
            else if(div(j-1,i).rem==0) node[m+1]=node[m-2];
            else node[m+1]=node[m-2]+1;
            
            node[m+2]=n+1;
            node[m+3]=n+2;
            if(j==1){
                i1=node[m+1];
                k1=node[m+2];
            }
        }
        node[m+1]=i1;
        node[m+3]=k1;
        i1=k1;
    }
    
    
    m1=m;
    n1=n;
    j1=nc+3;
    for(i=2;i<=nr1;i++){
        ntheta=nc*(i-1);
        for(j=1;j<=ntheta;j++){
            m+=3;
            n++;

            if(j==1) node[m+2]=j1;
            else if(div(j-1,i-1).rem==0) node[m+2]=node[m-1]+2;
            else node[m+2]=node[m-1]+1;
            
            node[m+1]=n-n1+1;
            node[m+3]=node[m+1]+1;
            if(j==1) k1=node[m+1];
        }
        node[m+3]=k1;
        j1=node[m+2]+2;
    }


    xv[1]=0;
    yv[1]=0;
    i1=1;
    for(i=1;i<=nr1;i++){
        sc=double(i)/double(nr1);
        ntheta=nc*i;
        for(j=1;j<=ntheta;j++){
            i1++;
            theta=2.0*pi*double(j-1)/double(ntheta);
            a=sin(theta);
            b=cos(theta);
            r=sc*x0*y0/sqrt(x0*x0*a*a+y0*y0*b*b);
            xv[i1]=r*b;
            yv[i1]=r*a;
        }
    }

    
    for (i=1;i<=nn;i++){
        iconv[i]=0;
        const1[i]=0;
        const2[i]=0.0;
    }

    
    i1=nc*(nr-1);
    n=nc*(nr-1)*(nr-2)/2+1;
    for(i=1;i<=i1;i++){
        n++;
        const1[n]=1;
        const2[n]=0.0;
    }

    for(i=1;i<=ne;i++){
        Q[i]=10.0;
    }

	kx=1.0;ky=1.0;
}


void bandwidth()//calculates the height of each individual column for
                //skyline format
{
    long i,j,num,n1,a;
    long n[npe+1];
    
    n1=nn+1;
    for(i=1;i<=n1;i++){
        maxh[i]=0;
    }
   
    for(num=1;num<=ne;num++){
        
        i=npe*(num-1);
        for(j=1;j<=npe;j++){
            n[j]=node[i+j];
        }
        
        for(i=1;i<=npe;i++){
            for(j=1;j<=npe;j++){
                if(n[i]>n[j]){
                    n1=n[i]+1;
                    a=n[i]-n[j]+1;
                }
                else{
                    n1=n[j]+1;
                    a=n[j]-n[i]+1;
                }
                if(maxh[n1]<a) maxh[n1]=a;
            }
        }
    }

    maxh[1]=1;
    maxh[2]=2;
    n1=nn+1;
    for(i=3;i<=n1;i++){
        maxh[i]+=maxh[i-1];
    }

}

void core_3nte(char type)//calculates element matrices and assembles them in 
                         //the global matrix for three node triangular elements
{
    long       i,j,ii,num;
    double     area,s,Tinfm,hm,qm,conv;
    long       n[npe+1];
    double     b[npe+1],c[npe+1],x[npe+1],y[npe+1];
    double     k1[npe+1][npe+1],k2[npe+1][npe+1],p1[npe+1],p2[npe+1],p3[npe+1];

    for (num=1;num<=ne;num++){

        for (i=1;i<=npe;i++){
            p1[i]=0.0;
            p2[i]=0.0;
            p3[i]=0.0;
            for(j=1;j<=npe;j++){
                k1[i][j]=0.0;
                k2[i][j]=0.0;
            }
        }

        i=npe*(num-1);
        for(j=1;j<=npe;j++){
            n[j]=node[i+j];
            x[j]=xv[n[j]];
            y[j]=yv[n[j]];
        }

        b[1]=y[3]-y[2];     b[2]=y[1]-y[3];     b[3]=y[2]-y[1];
        c[1]=x[2]-x[3];     c[2]=x[3]-x[1];     c[3]=x[1]-x[2];

        area=fabs( ((x[1]-x[3])*(y[2]-y[3])-(x[2]-x[3])*(y[1]-y[3])) /2.0);

        for(i=1;i<=npe;i++){
            for(j=i;j<=npe;j++){
                k1[i][j]=(kx*b[i]*b[j]+ky*c[i]*c[j])/(4.0*area);
            }
        }
        
        for (i=1;i<=npe;i++){
            p1[i]=Q[num]*area/3.0;
        }


        if ((iconv[n[1]]+iconv[n[2]])==2){
                s=sqrt( (x[1]-x[2])*(x[1]-x[2])+(y[1]-y[2])*(y[1]-y[2]) );
                Tinfm=(Tinf[n[1]]+Tinf[n[2]])/2;
                hm=(h[n[1]]+h[n[2]])/2;
                qm=(q[n[1]]+q[n[2]])/2;
                conv=hm*s/6.0;

                k2[1][1]=2.0*conv;
                k2[1][2]=conv;
                k2[2][2]=k2[1][1];

                p2[1]=hm*Tinfm*s/2.0;
                p2[2]=p2[1];

                p3[1]=qm*s/2.0;
                p3[2]=p3[1];
        }
        
        if ((iconv[n[2]]+iconv[n[3]])==2){
                s=sqrt( (x[2]-x[3])*(x[2]-x[3])+(y[2]-y[3])*(y[2]-y[3]) );
                Tinfm=(Tinf[n[2]]+Tinf[n[3]])/2;
                hm=(h[n[2]]+h[n[3]])/2;
                qm=(q[n[2]]+q[n[3]])/2;
                conv=hm*s/6.0;

                k2[2][2]=2.0*conv;
                k2[2][3]=conv;
                k2[3][3]=k2[2][2];

                p2[2]=hm*Tinfm*s/2.0;
                p2[3]=p2[2];

                p3[2]=qm*s/2.0;
                p3[3]=p3[2];
        }
        
        if ((iconv[n[3]]+iconv[n[1]])==2){
                s=sqrt( (x[3]-x[1])*(x[3]-x[1])+(y[3]-y[1])*(y[3]-y[1]) );
                Tinfm=(Tinf[n[3]]+Tinf[n[1]])/2;
                hm=(h[n[3]]+h[n[1]])/2;
                qm=(q[n[3]]+q[n[1]])/2;
                conv=hm*s/6.0;

                k2[1][1]=2.0*conv;
                k2[1][3]=conv;
                k2[3][3]=k2[1][1];

                p2[1]=hm*Tinfm*s/2.0;
                p2[3]=p2[1];

                p3[1]=qm*s/2.0;
                p3[3]=p3[1];
        }


        switch(type){
            case 'b' :
                for(i=1;i<=npe;i++){
                    p[n[i]]+=p1[i]+p2[i]+p3[i];
                    for(j=i;j<=npe;j++){
                        if(n[i]>n[j]) ii=(n[j]-1)*nb+n[i]-n[j]+1;
                        else          ii=(n[i]-1)*nb+n[j]-n[i]+1;
                        gk[ii]+=k1[i][j]+k2[i][j];
                    }
                }
                break;
            case 's' :
                for(i=1;i<=npe;i++){
                    p[n[i]]+=p1[i]+p2[i]+p3[i];
                    for(j=i;j<=npe;j++){
                        if(n[i]>n[j]) ii=maxh[n[i]]+n[i]-n[j];
                        else          ii=maxh[n[j]]+n[j]-n[i];
                        gk[ii]+=k1[i][j]+k2[i][j];
                    }
                }
                break;
        }
    }

}

void core_6nte(char type)//calculates element matrices and assembles them in
                         //the global matrix for six node triangular elements
{
    long       i,j,ii,num;
    double     J11,J12,J21,J22,area,a11,a12,a22,s,Tinfm,hm,qm,conv;
    long       n[npe+1];
    double     x[npe+1],y[npe+1],k1[npe+1][npe+1],k2[npe+1][npe+1];
	double     p1[npe+1],p2[npe+1],p3[npe+1];

    for (num=1;num<=ne;num++){

        for (i=1;i<=npe;i++){
            p1[i]=0.0;
            p2[i]=0.0;
            p3[i]=0.0;
            for(j=1;j<=npe;j++){
                k1[i][j]=0.0;
                k2[i][j]=0.0;
            }
        }

        i=npe*(num-1);
        for(j=1;j<=npe;j++){
            n[j]=node[i+j];
            x[j]=xv[n[j]];
            y[j]=yv[n[j]];
        }
        
        J11=x[1]-x[3];     J12=y[1]-y[3];
        J21=x[2]-x[3];     J22=y[2]-y[3];

        area=fabs( (J11*J22-J12*J21) /2.0);

        a11=(kx*J22*J22+ky*J21*J21)/2.0/area;
        a12=-(kx*J22*J12+ky*J21*J11)/2.0/area;
        a22=(kx*J12*J12+ky*J11*J11)/2.0/area;

        k1[1][1]=0.5*a11;
        k1[1][2]=-a12/6.0;
        k1[1][3]=(a11+a12)/6.0;
        k1[1][4]=2.0/3.0*a12;
        k1[1][5]=0.0;
        k1[1][6]=-2.0/3.0*(a11+a12);
        k1[2][2]=0.5*a22;
        k1[2][3]=(a12+a22)/6.0;
        k1[2][4]=2.0/3.0*a12;
        k1[2][5]=-2.0/3.0*(a12+a22);
        k1[2][6]=0.0;
        k1[3][3]=0.5*(a11+2*a12+a22);
        k1[3][4]=0.0;
        k1[3][5]=-2.0/3.0*(a12+a22);
        k1[3][6]=-2.0/3.0*(a11+a12);
        k1[4][4]=4.0/3.0*(a11+a12+a22);
        k1[4][5]=-4.0/3.0*(a11+a12);
        k1[4][6]=-4.0/3.0*(a12+a22);
        k1[5][5]=4.0/3.0*(a11+a12+a22);
        k1[5][6]=4.0/3.0*a12;
        k1[6][6]=4.0/3.0*(a11+a12+a22);

        for (i=4;i<=npe;i++){
            p1[i]=Q[num]*area/3.0;
        }

        if ((iconv[n[1]]+iconv[n[2]])==2){
                s=sqrt( (x[1]-x[2])*(x[1]-x[2])+(y[1]-y[2])*(y[1]-y[2]) );
                Tinfm=(Tinf[n[1]]+Tinf[n[4]]+Tinf[n[2]])/2;
                hm=(h[n[1]]+h[n[4]]+h[n[2]])/2;
                qm=(q[n[1]]+q[n[4]]+q[n[2]])/2;
                conv=hm*s/30.0;
                                
                k2[1][1]=4.0*conv;
                k2[1][2]=-conv;
                k2[1][4]=2.0*conv;
                k2[2][2]=k2[1][1];
                k2[2][4]=k2[1][4];
                k2[4][4]=16.0*conv;
                
                p2[1]=hm*Tinfm*s/6.0;
                p2[2]=p2[1];
                p2[4]=2.0*p2[1];

                p3[4]=qm*s/3.0;
        }

        if ((iconv[n[2]]+iconv[n[3]])==2){
                s=sqrt( (x[2]-x[3])*(x[2]-x[3])+(y[2]-y[3])*(y[2]-y[3]) );
                Tinfm=(Tinf[n[2]]+Tinf[n[5]]+Tinf[n[3]])/2;
                hm=(h[n[2]]+h[n[5]]+h[n[3]])/2;
                qm=(q[n[2]]+q[n[5]]+q[n[3]])/2;
                conv=hm*s/30.0;
                                
                k2[2][2]=4.0*conv;
                k2[2][3]=-conv;
                k2[2][5]=2.0*conv;
                k2[3][3]=k2[2][2];
                k2[3][5]=k2[2][5];
                k2[5][5]=16.0*conv;
                
                p2[2]=hm*Tinfm*s/6.0;
                p2[3]=p2[2];
                p2[5]=2.0*p2[2];

                p3[5]=qm*s/3.0;
        }

        if ((iconv[n[3]]+iconv[n[1]])==2){
                s=sqrt( (x[3]-x[1])*(x[3]-x[1])+(y[3]-y[1])*(y[3]-y[1]) );
                Tinfm=(Tinf[n[3]]+Tinf[n[6]]+Tinf[n[1]])/2;
                hm=(h[n[3]]+h[n[6]]+h[n[1]])/2;
                qm=(q[n[3]]+q[n[6]]+q[n[1]])/2;
                conv=hm*s/30.0;
                                
                k2[1][1]=4.0*conv;
                k2[1][3]=-conv;
                k2[1][6]=2.0*conv;
                k2[3][3]=k2[1][1];
                k2[3][6]=k2[1][6];
                k2[6][6]=16.0*conv;
                
                p2[1]=hm*Tinfm*s/6.0;
                p2[3]=p2[1];
                p2[6]=2.0*p2[1];

                p3[6]=qm*s/3.0;
        }
        

        switch(type){
            case 'b' :
                for(i=1;i<=npe;i++){
                    p[n[i]]+=p1[i]+p2[i]+p3[i];
                    for(j=i;j<=npe;j++){
                        if(n[i]>n[j]) ii=(n[j]-1)*nb+n[i]-n[j]+1;
                        else          ii=(n[i]-1)*nb+n[j]-n[i]+1;
                        gk[ii]+=k1[i][j]+k2[i][j];
                    }
                }
                break;
            case 's' :
                for(i=1;i<=npe;i++){
                    p[n[i]]+=p1[i]+p2[i]+p3[i];
                    for(j=i;j<=npe;j++){
                        if(n[i]>n[j]) ii=maxh[n[i]]+n[i]-n[j];
                        else          ii=maxh[n[j]]+n[j]-n[i];
                        gk[ii]+=k1[i][j]+k2[i][j];
                    }
                }
                break;
        }
    }

}

void boundary_sym_banded(long neqns,long nbw,double mat[],double rhs[],
			 long cond1[],double cond2[])
//imposes the boundary conditions on primary variables for
//a banded symmetric form
{
    long ii,i1,i2,i3,j;
    
    for (ii=1;ii<=neqns;ii++){
        
        if (cond1[ii]!=0){
            i1=ii;
            i2=ii;
            i3=(ii-1)*nbw;
            for (j=2;j<=nbw;j++){
               i1--;
               i2++;
               if(i1>=1){
                   rhs[i1]-=mat[(i1-1)*nbw+j]*cond2[ii];
                   mat[(i1-1)*nbw+j]=0.0;
               }
               if(i2<=neqns){
                   rhs[i2]-=mat[i3+j]*cond2[ii];
                   mat[i3+j]=0.0;
               }
            }
      
            rhs[ii]=cond2[ii];
            mat[i3+1]=1.0;
        }
    }

}

void boundary_sym_skyline(long neqns,long nbw,double mat[],double rhs[],
			  long maxh[],long cond1[],double cond2[])
//imposes the boundary conditions on primary variables for
//a skyline symmetric form
{
    long ii,i1,i2,j,j1,nbw1;

    nbw1=nbw-1; 
    for (ii=1;ii<=neqns;ii++){
        
        if (cond1[ii]!=0){
            i1=ii;
            i2=maxh[ii];
            j1=maxh[ii+1]-maxh[ii]-1;
            for(j=1;j<=j1;j++){
                i1--;
                rhs[i1]-=mat[i2+j]*cond2[ii];
                mat[i2+j]=0.0;
            }
            i1=ii;
            for(j=1;j<=nbw1;j++){
                i1++;
                if(i1<=neqns && j<=maxh[i1+1]-maxh[i1]-1){
                    rhs[i1]-=mat[maxh[i1]+j]*cond2[ii];
                    mat[maxh[i1]+j]=0.0;
                }
            }

            rhs[ii]=cond2[ii];
            mat[i2]=1.0;
        }
    }

}
void solve_gauss_sym_banded(long neqns,long nbw,double mat[],double rhs[])
//solves the system of equations in banded symmetric form by
//Gauss elimination method 
{

    long        i,j,ii,jj,m,n,n1,n2,n3;
    double      factor;
  
    m=neqns-1;
    for(n=1;n<=m;n++){
        n1=n+1;
        n2=n+nbw-1;
        n3=(n-1)*nbw;
        if (n2>neqns) n2=neqns;
        for(i=n1;i<=n2;i++){
            j=i-n+1;
            factor=mat[n3+j]/mat[n3+1];
            for(j=i;j<=n2;j++){
                ii=j-i+1;
                jj=j-n+1;
                mat[(i-1)*nbw+ii]-=factor*mat[n3+jj];
            }
            rhs[i]-=factor*rhs[n];
        }
    }


    // Back substitution

    for(n=neqns;n>=2;n--){
        n1=n-1;
        n2=n-nbw+1;
        rhs[n]/=mat[(n-1)*nb+1];
        if (n2<1) n2=1;
        for(i=n1;i>=n2;i--){
            j=n-i+1;
            rhs[i]-=mat[(i-1)*nbw+j]*rhs[n];
        }
    }
    
    rhs[1]/=mat[1];
}

void solve_LUdecom_sym_skyline(long neqns,double mat[],double rhs[],long maxh[])
//solves the system of equations in skyline symmetric form by
//LU decomposition method
{
    long   n,m,m1,m2,m3,m4,i,i1,i2,i3,ii,j,k;
    double a,b;

    for(n=1;n<=neqns;n++){
        m=maxh[n];
        m1=m+1;
        m2=maxh[n+1]-1;
        m3=m2-m1;
        if(m3>0){
            i=n-m3;
            i1=0;
            m4=m2;
            for(j=1;j<=m3;j++){
                i1++;
                m4--;
                i2=maxh[i];
                i3=maxh[i+1]-i2-1;
                if(i3>0){
                    ii=min(i1,i3);
                    a=0.0;
                    for(k=1;k<=ii;k++){
                        a+=mat[i2+k]*mat[m4+k];
                    }
                    mat[m4]-=a;
                }
                i++;
            }
        }
        if(m3>=0){
            i=n;
            b=0.0;
            for(ii=m1;ii<=m2;ii++){
                i--;
                i2=maxh[i];
                a=mat[ii]/mat[i2];
                b+=a*mat[ii];
                mat[ii]=a;
            }
            mat[m]-=b;
        }
    }

    //Reduation of right-hand-side veator

    for(n=1;n<=neqns;n++){
        m1=maxh[n]+1;
        m2=maxh[n+1]-1;
        if(m2-m1>=0){
            i=n;
            a=0.0;
            for(ii=m1;ii<=m2;ii++){
                i--;
                a+=mat[ii]*rhs[i];
            }
            rhs[n]-=a;
        }
    }

    //back substitution

    for(n=1;n<=neqns;n++){
        i=maxh[n];
        rhs[n]/=mat[i];
    }
    n=neqns;
    for(k=2;k<=neqns;k++){
        m1=maxh[n]+1;
        m2=maxh[n+1]-1;
        if(m2-m1>=0){
            i=n;
            for(ii=m1;ii<=m2;ii++){
                i--;
                rhs[i]-=mat[ii]*rhs[n];
            }
        }
        n--;
    }
}


void file_out()
//writes the results on some files
{
    long i,n;
    n=npe*ne;
    
    ofstream out1;
    out1.open("out1.dat");
    for (i=1;i<=nn;i++){
        out1<<p[i]<<"  ";
    }
    
    ofstream out2;
    out2.open("out2.dat");
    for (i=1;i<=n;i++){
        out2<<node[i]<<"  ";
    }
    
    ofstream out3;
    out3.open("out3.dat");
    for (i=1;i<=nn;i++){
        out3<<xv[i]<<"  ";
    }

    ofstream out4;
    out4.open("out4.dat");
    for (i=1;i<=nn;i++){
        out4<<yv[i]<<"  ";
    }

}
